// 引入必要接口
import { defineComponent, onMounted, ref } from 'vue';
import { GameUI, SetupComponentOptions, UIComponentProps,} from '@motajs/system-ui';
import { DefaultProps, onTick } from '@motajs/render-vue';
import { transitioned, useKey } from '../use';
import { linear, sleep } from 'mutate-animate';
import { gameKey } from '@motajs/system-action';
import { KeyCode } from '@motajs/client-base';
import { clamp } from 'lodash-es';

gameKey
    // 将后面注册的内容形成一个组，在修改快捷键时比较直观
    // 命名建议为 @ui_[UI 名称]
    .group('@ui_myUI', '游戏操作')
    .register({
        // 命名时，建议使用 @ui_[UI 名称]_[按键名称] 的格式
        id: '@ui_myUI_moveUp',
        // 在自定义快捷键界面显示的名称
        name: '上移',
        // 默认按键
        defaults: KeyCode.UpArrow
    })
    .register({
        id: '@ui_myUI_moveDown',
        name: '下移',
        defaults: KeyCode.DownArrow
    })
    .register({
        id: '@ui_myUI_moveLeft',
        name: '左移',
        defaults: KeyCode.LeftArrow
    })
    .register({
        id: '@ui_myUI_moveRight',
        name: '右移',
        defaults: KeyCode.RightArrow
    })
    .register({
        id: '@ui_myUI_slowMove', // 添加减速按键
        name: '减速移动',
        defaults: KeyCode.KeyZ
    });
;

// 定义组件的参数
export interface MyComProps extends UIComponentProps, DefaultProps {}

// 定义组件的参数，需要传递给 vue
const myComProps = {
    // 这两个参数不能少
    props: ['controller', 'instance']
} satisfies SetupComponentOptions<MyComProps>;

enum GameState {
    PLAYING = 'playing',
    VICTORY = 'victory',
    DEFEAT = 'defeat'
}
const heroX = ref(388)  //玩家坐标
const heroY = ref(304)
const heroV = ref(0.18)  //玩家速度
const downingW = ref(0)  //记录按键是否被按下
const downingS = ref(0)
const downingA = ref(0)
const downingD = ref(0)
const slowing = ref(0)
let lastTime: number | null = null;
let deltaTime: number | null = null;
const gameState = ref(GameState.PLAYING);
const bossX = ref(388);
    const bossY = ref(-50); // 初始位置在屏幕上方
    const bossSize = 64; // Boss图标大小
    const bossSpeed = 0.1;
    const bossTargetY = 50; // Boss停止的位置
    const bossHealth = ref(2000);
    const bossMaxHealth = 2000;
    const bossHasEntered = ref(false);
    let bossMoveTime = 0;
    
    // Boss子弹
interface BossBullet {
    id: number;
    x: number;
    y: number;
    active: boolean;
    vx?: number; // 水平速度
    vy?: number; // 垂直速度
}
    const bossBullets = ref<BossBullet[]>([]);
    const bossBulletSpeed = 0.2;
    const bossBulletSpawnRate = 500; // Boss发射子弹间隔
    let bossBulletIdCounter = 0;
    let lastBossBulletTime = 0;
    //玩家子弹
    interface Bullet {
        id: number;
        x: number;
        y: number;
        active: boolean;
    }
    const bullets = ref<Bullet[]>([]);  //子弹数组
    const bulletSpeed = 0.5;  //子弹速度
    const bulletSpawnRate = 200;  //子弹生成间隔(ms)
    let bulletIdCounter = 0;  //子弹ID计数器
    let lastBulletTime = 0;  //上次生成子弹的时间
    // 在GameState枚举后添加Boss状态
enum BossState {
    ENTERING = 'entering',
    PATTERN1 = 'pattern1', // 环形弹幕
    PATTERN2 = 'pattern2', // 扇形弹幕  
    PATTERN3 = 'pattern3', // 螺旋弹幕
    PATTERN4 = 'pattern4'  // 随机散射
}

// 添加Boss状态变量
const bossState = ref(BossState.ENTERING);
let bossPatternTimer = 0;
const patternDuration = 3000; // 每个模式持续时间



// 定义组件内容
export const MyCom = defineComponent<MyComProps>(props => {
    const selected = ref(0);
    const [key, scope] = useKey();
    const click = () => {
        if (gameState.value !== GameState.PLAYING) {
            // 重新开始游戏
            handleGameEndAction();
        }
    };
    // 定义动画属性
    const value = transitioned(
        -32, // 初始值
        500, // 动画时长 500ms
        linear() // 速率曲线
    )!;
    const isPageVisible = ref(true);

    // 页面可见性变化处理函数
    const handleVisibilityChange = () => {
    isPageVisible.value = !document.hidden;
    if (!isPageVisible.value) {
        // 页面不可见时，重置时间戳，避免恢复时产生巨大的deltaTime
        lastTime = null;
    }
    };
    // 在 UI 挂载完毕后执行
    onMounted(async () => {
        resetGame();
        document.addEventListener('visibilitychange', handleVisibilityChange);
        // 动画属性
        while (true) {
        value.set(0);
        await sleep(500);
        value.set(-32,0);
        }
    }
    );

    const spawnBullet = () => {
    const newBullet: Bullet = {
        id: bulletIdCounter++,
        x: heroX.value,
        y: heroY.value,
        active: true
    };
    bullets.value.push(newBullet);
    };
    const updateBullets = (deltaTime: number) => {
    bullets.value = bullets.value.map(bullet => {
        if (!bullet.active) return bullet;
        
        // 子弹向上移动
        bullet.y -= bulletSpeed * deltaTime;
        
        // 检查子弹是否离开屏幕（假设屏幕高度为416）
        if (bullet.y < -20) {
            bullet.active = false;
        }
        
        return bullet;
    }).filter(bullet => bullet.active);
    };
    // 生成Boss子弹
    // 环形弹幕
const spawnCirclePattern = (bulletCount: number, speed: number) => {
    for (let i = 0; i < bulletCount; i++) {
        const angle = (i / bulletCount) * Math.PI * 2;
        const newBullet: BossBullet = {
            id: bossBulletIdCounter++,
            x: bossX.value,
            y: bossY.value + bossSize/2,
            active: true,
            vx: Math.cos(angle) * speed,
            vy: Math.sin(angle) * speed
        };
        bossBullets.value.push(newBullet);
    }
};

// 扇形弹幕（瞄准玩家）
const spawnFanPattern = (bulletCount: number, spreadAngle: number, speed: number) => {
    // 计算玩家方向
    const dx = heroX.value - bossX.value;
    const dy = heroY.value - bossY.value;
    const baseAngle = Math.atan2(dy, dx);
    
    for (let i = 0; i < bulletCount; i++) {
        const angle = baseAngle - spreadAngle/2 + (spreadAngle * i) / (bulletCount - 1);
        const newBullet: BossBullet = {
            id: bossBulletIdCounter++,
            x: bossX.value,
            y: bossY.value + bossSize/2,
            active: true,
            vx: Math.cos(angle) * speed,
            vy: Math.sin(angle) * speed
        };
        bossBullets.value.push(newBullet);
    }
};

// 螺旋弹幕
let spiralAngle = 0;
const spawnSpiralPattern = (speed: number) => {
    const newBullet: BossBullet = {
        id: bossBulletIdCounter++,
        x: bossX.value,
        y: bossY.value + bossSize/2,
        active: true,
        vx: Math.cos(spiralAngle) * speed,
        vy: Math.sin(spiralAngle) * speed
    };
    bossBullets.value.push(newBullet);
    spiralAngle += 0.3; // 每次增加角度，形成螺旋
};

// 随机散射
const spawnRandomPattern = (bulletCount: number, speed: number) => {
    for (let i = 0; i < bulletCount; i++) {
        const angle = Math.random() * Math.PI * 2;
        const newBullet: BossBullet = {
            id: bossBulletIdCounter++,
            x: bossX.value,
            y: bossY.value + bossSize/2,
            active: true,
            vx: Math.cos(angle) * speed,
            vy: Math.sin(angle) * speed
        };
        bossBullets.value.push(newBullet);
    }
};
    // 更新Boss子弹位置
    const updateBossBullets = (deltaTime: number) => {
    bossBullets.value = bossBullets.value.map(bullet => {
        if (!bullet.active) return bullet;
        
        // 使用速度向量移动子弹
        if (bullet.vx !== undefined && bullet.vy !== undefined) {
            bullet.x += bullet.vx * deltaTime;
            bullet.y += bullet.vy * deltaTime;
        } else {
            // 向下移动（兼容旧子弹）
            bullet.y += bossBulletSpeed * deltaTime;
        }
        
        // 检查子弹是否离开屏幕（所有边界）
        if (bullet.y > 436 || bullet.y < -20 || bullet.x < 150 || bullet.x > 646) {
            bullet.active = false;
        }
        
        return bullet;
    }).filter(bullet => bullet.active);
};

const resetGame = () => {

        // 重置玩家位置
        heroX.value = 388;
        heroY.value = 304;
        
        // 重置Boss
        bossX.value = 388;
        bossY.value = -100;
        bossHealth.value = bossMaxHealth;
        bossHasEntered.value = false;
        
        // 清空子弹
        bullets.value = [];
        bossBullets.value = [];
        
        // 重置计数器
        bulletIdCounter = 0;
        bossBulletIdCounter = 0;
        lastBulletTime = 0;
        lastBossBulletTime = 0;
        downingW.value = 0;
        downingS.value = 0;
        downingA.value = 0;
        downingD.value = 0;
        slowing.value = 0;
        lastTime = null;

        // 重置Boss状态
    bossState.value = BossState.ENTERING;
    bossPatternTimer = 0;
    spiralAngle = 0;
    bossMoveTime = 0;
        
        // 重置游戏状态
        core.status.hero.hp = 10000000
        gameState.value = GameState.PLAYING;
    };

    const handleGameEndAction = () => {
    if (gameState.value === GameState.VICTORY) {
        props.controller.close(props.instance);
        // 胜利时关闭UI并触发事件
        core.insertCommonEvent('胜利');
    } else if (gameState.value === GameState.DEFEAT) {
        // 失败时重置游戏
        resetGame();
    }
};
    // 碰撞检测函数
    const checkCollisions = () => {
        // 玩家子弹与Boss碰撞检测（圆形与圆形）
        bullets.value.forEach(bullet => {
            if (!bullet.active) return;
            
            const distanceX = bullet.x - bossX.value;
            const distanceY = bullet.y - bossY.value;
            const distanceSquared = distanceX * distanceX + distanceY * distanceY;
            
            // 子弹半径5，Boss半径32（64/2）
            if (distanceSquared < (5 + 32) * (5 + 32)) {
                bullet.active = false;
                bossHealth.value -= 10;
                
                // 检查Boss是否被击败
                if (bossHealth.value <= 0) {
                    bossHealth.value = 0;
                    gameState.value = GameState.VICTORY;
                    bullets.value = [];
                    bossBullets.value = [];
                }
            }
        });
        
        // 玩家与Boss子弹碰撞检测（圆形与圆形）
        bossBullets.value.forEach(bullet => {
            if (!bullet.active) return;
            
            const distanceX = heroX.value - bullet.x;
            const distanceY = heroY.value - bullet.y;
            const distanceSquared = distanceX * distanceX + distanceY * distanceY;
            
            // 玩家半径5，子弹半径5
            if (distanceSquared < (5 + 5) * (5 + 5)) {
                bullet.active = false;
                gameState.value = GameState.DEFEAT;
                bullets.value = [];
                bossBullets.value = [];
            }
        });
        
        // 玩家与Boss碰撞检测
        let distanceX = heroX.value - bossX.value;
        let distanceY = heroY.value - bossY.value;
        if(distanceX<0){distanceX=-distanceX}
        if(distanceY<0){distanceY=-distanceY}
        if (distanceX<35 && distanceY<35) {
            gameState.value = GameState.DEFEAT;
            bullets.value = [];
            bossBullets.value = [];
        }
    };

    //按键操作
    key.realize('@ui_myUI_moveUp', () => {
        if (gameState.value !== GameState.PLAYING) {
        handleGameEndAction();
        }
        downingW.value = 1
    },{ type: 'down' } );
    key.realize('@ui_myUI_moveUp', () => {
        downingW.value = 0
    });
    key.realize('@ui_myUI_moveDown', () => {
        if (gameState.value !== GameState.PLAYING) {
        handleGameEndAction();
        }
        downingS.value = 1
    },{ type: 'down' });
    key.realize('@ui_myUI_moveDown', () => {
        downingS.value = 0
    });
    key.realize('@ui_myUI_moveLeft', () => {
        if (gameState.value !== GameState.PLAYING) {
        handleGameEndAction();
        }
        downingA.value = 1
    },{ type: 'down' });
    key.realize('@ui_myUI_moveLeft', () => {
        downingA.value = 0
    });
    key.realize('@ui_myUI_moveRight', () => {
        if (gameState.value !== GameState.PLAYING) {
        handleGameEndAction();
        }
        downingD.value = 1
    },{ type: 'down' });
    key.realize('@ui_myUI_moveRight', () => {
        downingD.value = 0
    });
    key.realize('@ui_myUI_slowMove', () => {
        if (gameState.value !== GameState.PLAYING) {
        resetGame();
        }
    slowing.value = 1
    }, { type: 'down' });
    key.realize('@ui_myUI_slowMove', () => {
        slowing.value = 0
    });
    onTick(time => {
        if (gameState.value !== GameState.PLAYING) {
            return;
        }
        //console.log('当前帧时间:', time);
        if (lastTime === null) {
            lastTime = time;
        }
        deltaTime = time-lastTime
        lastTime = time
        core.status.hero.hp -= deltaTime
        if(core.status.hero.hp<1){core.status.hero.hp=1}
        //console.log(core.status.hero.hp)
        const maxDeltaTime = 100; // 最大100ms
        if (deltaTime > maxDeltaTime) {
            deltaTime = maxDeltaTime;
        }
        const currentSpeed = slowing.value === 1 ? heroV.value * 0.5 : heroV.value;
        if(downingW.value===1){
            heroY.value-=currentSpeed*deltaTime
            heroY.value=clamp(heroY.value,20,416)
        }
        if(downingS.value===1){
            heroY.value+=currentSpeed*deltaTime
            heroY.value=clamp(heroY.value,20,416)
        }
        if(downingA.value===1){
            heroX.value-=currentSpeed*deltaTime
            heroX.value=clamp(heroX.value,180,596)
        }
        if(downingD.value===1){
            heroX.value+=currentSpeed*deltaTime
            heroX.value=clamp(heroX.value,180,596)
        }
        if (deltaTime) {  //生成玩家子弹
            lastBulletTime += deltaTime;
            if (lastBulletTime >= bulletSpawnRate) {
                spawnBullet();
                lastBulletTime = 0;
            }
        }
        if (deltaTime) {  //更新玩家子弹
            updateBullets(deltaTime);
            //console.log(bullets);
        }
        // 更新Boss位置
if (!bossHasEntered.value) {
    bossY.value += bossSpeed * deltaTime;
    if (bossY.value >= bossTargetY) {
        bossY.value = bossTargetY;
        bossHasEntered.value = true;
        bossState.value = BossState.PATTERN1;
        // 重置移动相关变量，确保从中心开始移动
        bossMoveTime = 0;
    }
} else {
    // Boss轻微左右移动 - 从当前位置开始
    bossMoveTime += deltaTime;
    bossX.value = 388 + Math.sin(bossMoveTime * 0.001) * 100; // 从中心388开始摆动
    bossX.value = clamp(bossX.value, 250, 526);
    
    // 弹幕模式切换
    bossPatternTimer += deltaTime;
    if (bossPatternTimer >= patternDuration) {
        switch (bossState.value) {
            case BossState.PATTERN1:
                bossState.value = BossState.PATTERN2;
                break;
            case BossState.PATTERN2:
                bossState.value = BossState.PATTERN3;
                break;
            case BossState.PATTERN3:
                bossState.value = BossState.PATTERN4;
                break;
            case BossState.PATTERN4:
                bossState.value = BossState.PATTERN1;
                break;
        }
        bossPatternTimer = 0;
    }
}
        // 生成Boss子弹（根据当前弹幕模式）
if (bossHasEntered.value && deltaTime) {
    lastBossBulletTime += deltaTime;
    
    let spawnRate = bossBulletSpawnRate;
    
    switch (bossState.value) {
        case BossState.PATTERN1: // 环形弹幕
            spawnRate = 350;
            if (lastBossBulletTime >= spawnRate) {
                spawnCirclePattern(8, 0.15); // 8方向环形
                lastBossBulletTime = 0;
            }
            break;
            
        case BossState.PATTERN2: // 扇形弹幕
            spawnRate = 300;
            if (lastBossBulletTime >= spawnRate) {
                spawnFanPattern(5, Math.PI/3, 0.2); // 5发60度扇形
                lastBossBulletTime = 0;
            }
            break;
            
        case BossState.PATTERN3: // 螺旋弹幕
            spawnRate = 10;
            if (lastBossBulletTime >= spawnRate) {
                spawnSpiralPattern(0.2); // 螺旋发射
                lastBossBulletTime = 0;
            }
            break;
            
        case BossState.PATTERN4: // 随机散射
            spawnRate = 30;
            if (lastBossBulletTime >= spawnRate) {
                spawnRandomPattern(3, 0.18); // 3发随机方向
                lastBossBulletTime = 0;
            }
            break;
    }
}
        
        // 更新子弹位置
        if (deltaTime) {
            updateBullets(deltaTime);
            updateBossBullets(deltaTime);
        }
        
        // 碰撞检测
        checkCollisions();
    });
    // 获取注册的图片
    const gamebg = core.material.images.images['gamebg.webp'];
    let showpoint=0
    if(core.getFlag('showpoint')===1){
        showpoint=5
    }
    // 在这里编写你的 UI 即可
    return () => (
    <container loc={[0, 0, 596, 416, 0, 0]} onClick={click}>
        <image image={gamebg} loc={[180, value.ref.value, void 0, void 0, 0, 0]}/>
        <icon icon="goldHornSlime" loc={[heroX.value, heroY.value,32,32,0.5,0.5]} animate />
        <g-circle circle={[heroX.value, heroY.value, showpoint]} fillStyle="#f99a9aff" />
        {bullets.value.map(bullet => (
                <g-circle 
                    key={bullet.id}
                    circle={[bullet.x, bullet.y, 5]} // 圆心在子弹坐标
                    fillStyle="#ffff00" // 填充颜色为黄色
                />
            ))}
        {/* Boss */}
            <icon icon="rock" loc={[bossX.value, bossY.value, bossSize, bossSize, 0.5, 0.5]} animate />
        {/* Boss子弹 */}
            {bossBullets.value.map(bullet => (
                <g-circle 
                    key={bullet.id}
                    circle={[bullet.x, bullet.y, 5]}
                    fillStyle="#ff0000"
                />
            ))}
        {/* Boss血条 */}
            {bossHasEntered.value && (
                <container loc={[180, 0, 416, 20, 0, 0]}>
                    {/* 血条背景 - 使用黑色矩形 */}
                    <g-rect loc={[0, 0, 416, 20]} fillStyle="#333" />
                    {/* 血量 - 使用红色矩形 */}
                    <g-rect loc={[0, 0, (bossHealth.value / bossMaxHealth) * 416, 20]} fillStyle="#f00" />
                    {/* 血量文字 */}
                    <text 
                        text={`BOSS: ${bossHealth.value}/${bossMaxHealth}`} 
                        loc={[208, 10, void 0, void 0, 0.5, 0.5]} 
                        // 文字样式可能需要根据实际接口调整
                        // style={{ color: '#fff', fontSize: '12px' }}
                    />
                </container>
            )}
        {/* 游戏结束提示 */}
            {gameState.value === GameState.VICTORY && (
                <container loc={[0, 0, 596, 416, 0, 0]}>
                    {/* 半透明遮罩 - 使用半透明黑色矩形 */}
                    <g-rect loc={[0, 0, 596, 416]} fillStyle="rgba(0,0,0,0.7)" />
                    <text 
                        text="游戏胜利！按任意键提交成绩" 
                        loc={[298, 208, void 0, void 0, 0.5, 0.5]} 
                        // 文字样式可能需要根据实际接口调整
                        // style={{ color: '#fff', fontSize: '24px' }}
                    />
                </container>
            )}
            
            {gameState.value === GameState.DEFEAT && (
                <container loc={[0, 0, 596, 416, 0, 0]}>
                    {/* 半透明遮罩 - 使用半透明黑色矩形 */}
                    <g-rect loc={[0, 0, 596, 416]} fillStyle="rgba(0,0,0,0.7)" />
                    <text 
                        text="游戏失败！按任意键重新开始" 
                        loc={[298, 208, void 0, void 0, 0.5, 0.5]} 
                        // 文字样式可能需要根据实际接口调整
                        // style={{ color: '#fff', fontSize: '24px' }}
                    />
                </container>
            )}
    </container>
);
}, myComProps);

// 定义 UI 对象
export const MyUI = new GameUI('my-ui', MyCom);